//
//  TodoItemDetail.swift
//  Do It
//
//  Created by Jim Dovey on 5/20/20.
//  Copyright © 2020 Pragmatic Programmers. All rights reserved.
//

import SwiftUI

// START:EmptyTodoItem
fileprivate extension TodoItem {
    static var empty = TodoItem(title: "", priority: .normal,
                                listID: .null)
}
// END:EmptyTodoItem

struct TodoItemDetail: View {
    let item: TodoItem
    @EnvironmentObject private var data: DataCenter
    // START:EditableState
    @State private var editingItem: TodoItem = .empty
    @State private var showingEditor = false
    // END:EditableState
    
    // START:AddEditButton
    var body: some View {
        VStack(alignment: .leading) {
            // <literal:elide> view content </literal:elide>
            // END:AddEditButton
            Rectangle()
                .fill(data.list(for: item).color.uiColor)
                .edgesIgnoringSafeArea(.top)
                .frame(height: 210)
                .overlay(TitleOverlay(item: item))
            
            if item.notes != nil {
                Text(item.notes!)
                    .padding(.horizontal)
            }
            
            Spacer()
            // START:AddEditButton
        }
        .navigationBarTitle("", displayMode: .inline)
        // START_HIGHLIGHT
        .navigationBarItems(trailing: editButton)
        .sheet(isPresented: $showingEditor) {
            self.editor.environmentObject(self.data)
        }
        // END_HIGHLIGHT
    }
    // END:AddEditButton
    
    // START:EditCancelButton
    private var cancelButton: some View {
        Button(action: {
            self.showingEditor.toggle() //<label id="code.ch4.detail.editor.toggle"/>
        }) {
            Text("Cancel")
                .foregroundColor(.accentColor)
        }
    }
    // END:EditCancelButton
    
    // START:EditDoneButton
    private var doneButton: some View {
        Button(action: {
            self.data.updateTodoItem(self.editingItem) //<label id="code.ch4.detail.editor.apply"/>
            self.showingEditor.toggle()
        }) {
            Text("Done")
                .bold()
                .foregroundColor(.accentColor)
        }
    }
    // END:EditDoneButton
    
    // START:EditorView
    private var editor: some View {
        NavigationView {
            TodoItemEditor(item: $editingItem)
            .navigationBarItems(leading: cancelButton,
                                trailing: doneButton)
        }
    }
    // END:EditorView
    
    // START:EditButton
    private var editButton: some View {
        Button(action: {
            self.editingItem = self.item
            self.showingEditor.toggle()
        }) {
            Image(systemName: "square.and.pencil")
                .imageScale(.large)
                .foregroundColor(.accentColor)
        }
        .accessibility(label: Text("Edit"))
    }
    // END:EditButton
    
    private struct TitleOverlay: View {
        let item: TodoItem
        
        var gradient: LinearGradient {
            LinearGradient(
                gradient: Gradient(colors: [
                    Color.black.opacity(0.6),
                    Color.black.opacity(0),
                ]),
                startPoint: .bottom,
                endPoint: .init(x: 0.5, y: 0.1))
        }
        
        var body: some View {
            ZStack(alignment: .bottomLeading) {
                Rectangle().fill(gradient)
                VStack(alignment: .leading, spacing: 8) {
                    Text(item.title)
                        .font(.title)
                        .bold()
                    HStack(alignment: .firstTextBaseline) {
                        Text("Priority: ") + Text(item.priority.rawValue.capitalized).bold()
                        Spacer()
                        if item.date != nil {
                            Text("\(item.date!, formatter: Formatters.mediumDate)")
                        }
                        else {
                            Text("No Due Date")
                        }
                    }
                    .font(.subheadline)
                }
                .foregroundColor(.white)
                .padding()
            }
        }
    }
}

struct TodoItemDetail_Previews: PreviewProvider {
    static var previews: some View {
        NavigationView {
            TodoItemDetail(item: defaultTodoItems[0])
                .environmentObject(DataCenter())
        }
    }
}
